import streamlit as st
import sqlite3
import json
from sqlite3 import Connection
import pandas as pd  # 新增，用于处理Excel文件

# 数据库连接函数
def get_db_connection() -> Connection:
    conn = sqlite3.connect('questions.db')
    return conn

# 初始化数据库表
def init_db():
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute('''
        CREATE TABLE IF NOT EXISTS questions (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            question TEXT,
            correct_answer TEXT,
            option_a TEXT,
            option_b TEXT,
            option_c TEXT,
            option_d TEXT
        )
    ''')
    conn.commit()
    conn.close()

# 添加题目
def add_question(question, correct_answer, option_a, option_b, option_c, option_d):
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute('''
        INSERT INTO questions (question, correct_answer, option_a, option_b, option_c, option_d)
        VALUES (?,?,?,?,?,?)
    ''', (question, correct_answer, option_a, option_b, option_c, option_d))
    conn.commit()
    conn.close()

# 删除题目
def delete_question(id):
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute('DELETE FROM questions WHERE id =?', (id,))
    conn.commit()
    conn.close()

# 更新题目
def update_question(id, question, correct_answer, option_a, option_b, option_c, option_d):
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute('''
        UPDATE questions
        SET question =?, correct_answer =?, option_a =?, option_b =?, option_c =?, option_d =?
        WHERE id =?
    ''', (question, correct_answer, option_a, option_b, option_c, option_d, id))
    conn.commit()
    conn.close()

# 查询所有题目
def get_all_questions():
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute('SELECT * FROM questions')
    questions = cursor.fetchall()
    conn.close()
    return questions

# 将数据库导出为JSON文件
def export_to_json():
    questions = get_all_questions()
    questions_dict = [
        {
            'id': q[0],
            'question': q[1],
            'correct_answer': q[2],
            'options': [
                {'code': 'A', 'option': q[3]},
                {'code': 'B', 'option': q[4]},
                {'code': 'C', 'option': q[5]},
                {'code': 'D', 'option': q[6]}
            ]
        }
        for q in questions
    ]
    with open('questions.json', 'w') as f:
        json.dump(questions_dict, f, ensure_ascii=False, indent=4)
    return 'questions.json'

# 新增函数，用于从Excel文件导入题目数据到数据库
def import_from_excel(file):
    try:
        df = pd.read_excel(file)  # 读取Excel文件内容到DataFrame
        if all(col in df.columns for col in ['question', 'correct_answer', 'option_a', 'option_b', 'option_c', 'option_d']):
            conn = get_db_connection()
            cursor = conn.cursor()
            for index, row in df.iterrows():
                question = row['question']
                correct_answer = row['correct_answer']
                option_a = row['option_a']
                option_b = row['option_b']
                option_c = row['option_c']
                option_d = row['option_d']
                add_question(question, correct_answer, option_a, option_b, option_c, option_d)
            conn.commit()
            conn.close()
            return True
        else:
            st.error("Excel文件列名不符合要求，请确保包含'question', 'correct_answer', 'option_a', 'option_b', 'option_c', 'option_d'这些列。")
            return False
    except Exception as e:
        st.error(f"导入Excel文件时出错: {str(e)}")
        return False


# 主程序
def main():
    st.title("轻量题库管理系统1.0版")
    
    # 初始化数据库
    init_db()
    
    # 选择操作
    operation = st.sidebar.selectbox("选择操作", ["查看题目", "添加题目", "更新题目", "删除题目", "导出为JSON", "导入Excel文件"])  # 修改这里，添加导入Excel文件选项
    
    if operation == "查看题目":
        st.header("所有题目")
        questions = get_all_questions()
        for q in questions:
            st.write(f"ID: {q[0]}")
            st.write(f"题目: {q[1]}")
            st.write(f"正确答案: {q[2]}")
            st.write(f"选项 A: {q[3]}")
            st.write(f"选项 B: {q[4]}")
            st.write(f"选项 C: {q[5]}")
            st.write(f"选项 D: {q[6]}")
            st.write("---")
    
    elif operation == "添加题目":
        st.header("添加题目")
        question = st.text_input("题目")
        correct_answer = st.text_input("正确答案")
        option_a = st.text_input("选项 A")
        option_b = st.text_input("选项 B")
        option_c = st.text_input("选项 C")
        option_d = st.text_input("选项 D")
        
        if st.button("提交"):
            add_question(question, correct_answer, option_a, option_b, option_c, option_d)
            st.success("题目添加成功！")
    
    elif operation == "更新题目":
        st.header("更新题目")
        questions = get_all_questions()
        question_ids = [q[0] for q in questions]
        selected_id = st.selectbox("选择要更新的题目 ID", question_ids)
        
        if selected_id:
            selected_question = next((q for q in questions if q[0] == selected_id), None)
            if selected_question:
                question = st.text_input("题目", value=selected_question[1])
                correct_answer = st.text_input("正确答案", value=selected_question[2])
                option_a = st.text_input("选项 A", value=selected_question[3])
                option_b = st.text_input("选项 B", value=selected_question[4])
                option_c = st.text_input("选项 C", value=selected_question[5])
                option_d = st.text_input("选项 D", value=selected_question[6])
                
                if st.button("更新"):
                    update_question(selected_id, question, correct_answer, option_a, option_b, option_c, option_d)
                    st.success("题目更新成功！")
    
    elif operation == "删除题目":
        st.header("删除题目")
        questions = get_all_questions()
        question_ids = [q[0] for q in questions]
        selected_id = st.selectbox("选择要删除的题目 ID", question_ids)
        
        if st.button("删除"):
            delete_question(selected_id)
            st.success("题目删除成功！")
    
    elif operation == "导出为JSON":
        st.header("导出为JSON")
        if st.button("导出"):
            json_file_path = export_to_json()
            st.success(f"导出成功！文件路径：{json_file_path}")
            st.download_button(
                label="下载 JSON 文件",
                data=open(json_file_path, 'rb').read(),
                file_name='questions.json',
                mime='application/json'
            )

    elif operation == "导入Excel文件":  # 新增的导入Excel文件的操作逻辑处理部分
        st.header("导入Excel文件")
        uploaded_file = st.file_uploader("上传Excel文件", type=["xlsx", "xls"])
        if uploaded_file is not None:
            if st.button("导入"):
                if import_from_excel(uploaded_file):
                    st.success("Excel文件导入成功！")
                else:
                    st.error("导入失败，请检查文件格式及内容是否正确。")


if __name__ == "__main__":
    main()